/* NVTV NV internal encoder -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_nx.c,v 1.5 2004/01/27 17:05:26 dthierbach Exp $
 *
 * Contents:
 *
 * Internal NV17 TV encoder ("Zoran" ??)
 *
 */

#include "local.h" /* before everything else */
#include "xfree.h" 

#include <stdio.h>
#include <string.h>

#include "mmio.h"
#include "bitmask.h"
#include "tv_nx.h"
#include "nv_type.h"

/* -------- Registers -------- */

/* Here's an attempt to guess the meaning of some of the TV registers.
 *
 * 00-03 subcarrier frequency (fsub/Hz * 159.07287 = value)
 * 0c    hue-related (phase offset?)
 * 0e    black to white height
 * 10    hsync height
 * 31    active pixels*2 (=1440)
 * 3c    vsync height 
 *
 * The following registers affect the hue, so they are either burst
 * timings or heights, or other horizontal timings that influence the burst:
 *
 * 08,09,0a,0b,0c
 *
 */

/* -------- TV indirect access -------- */ 

CARD32 
NVReadTV (NVPtr pNv, int reg)
{
  MMIO_OUT32 (pNv->riva.PTV, 0x220, reg);
  return MMIO_IN32 (pNv->riva.PTV, 0x224);
}

void 
NVWriteTV (NVPtr pNv, int reg, CARD32 val)
{
  MMIO_OUT32 (pNv->riva.PTV, 0x220, reg);
  MMIO_OUT32 (pNv->riva.PTV, 0x224, val);
}

void 
NVAndOrTV (NVPtr pNv, int reg, CARD32 mask, CARD32 val)
{
  CARD32 tmp;

  MMIO_OUT32 (pNv->riva.PTV, 0x220, reg);
  tmp = MMIO_IN32 (pNv->riva.PTV, 0x224);
  tmp &= mask;
  tmp |= val;
  MMIO_OUT32 (pNv->riva.PTV, 0x224, val);
}

/* -------- -------- */

static CARD32
TVNxSetFiltCoeff (int val) 
{
  if (val < 0) {
    return SetBitField (-val,8:0,16:8) | SetBit(31);
  } else {
    return SetBitField ( val,8:0,16:8);
  }
}

static int
TVNxGetFiltCoeff (CARD32 val) 
{
  if (val & SetBit(31)) {
    return - GetBF (val,16:8);
  } else {
    return GetBF (val,16:8);
  }
}

/* -------- -------- */

void TVNxCreate (TVEncoderObj *this, TVChip chip_type, void *ctrl)
{
  RAISE (MSG_DEBUG, "tv create nx");
  this->type = chip_type;
  this->ctrl = ctrl;
  this->hwconfig = 0;  
  this->hwstate = 0; 
}

void TVNxSetPort (TVEncoderObj *this, int port)
{
}

void TVNxGetPort (TVEncoderObj *this, int *port)
{
}

void TVNxInitRegs (TVEncoderObj *this, int port)
{
}

void 
TVNxSetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register NVPtr pNv = (NVPtr) this->ctrl;
  int i;

  RAISE (MSG_DEBUG, "tv set regs nx");

  NVWriteTV (pNv, 0x00, Set8Bits (r->nx.ind.fsub >> 24));
  NVWriteTV (pNv, 0x01, Set8Bits (r->nx.ind.fsub >> 16));
  NVWriteTV (pNv, 0x02, Set8Bits (r->nx.ind.fsub >> 8));
  NVWriteTV (pNv, 0x03, Set8Bits (r->nx.ind.fsub));
  NVWriteTV (pNv, 0x04, Set8Bits (r->nx.ind.nxreg04));
  NVWriteTV (pNv, 0x08, Set8Bits (r->nx.ind.nxreg08));
  NVWriteTV (pNv, 0x09, SetBitField (r->nx.ind.nxreg09, 6:0,6:0));
  NVWriteTV (pNv, 0x0a, Set8Bits (r->nx.ind.nxreg0a));
  NVWriteTV (pNv, 0x0b, Set8Bits (r->nx.ind.nxreg0b));
  NVWriteTV (pNv, 0x0c, Set8Bits (r->nx.ind.nxreg0c));
  NVWriteTV (pNv, 0x0e, SetBitField (r->nx.ind.nxreg0e,9:2,7:0));
  NVWriteTV (pNv, 0x10, SetBitField (r->nx.ind.nxreg10,9:2,7:0));
  NVWriteTV (pNv, 0x17, SetBitField (r->nx.ind.nxreg17,9:2,7:0));
  NVWriteTV (pNv, 0x19, 0x00); /* MV off for now */
  NVWriteTV (pNv, 0x1e, SetBitField (r->nx.ind.nxreg1e,9:2,7:0));
  NVWriteTV (pNv, 0x20, Set8Bits (r->nx.ind.nxreg20));
  NVWriteTV (pNv, 0x22, Set8Bits (r->nx.ind.nxreg22));
  NVWriteTV (pNv, 0x29, SetBitField (r->nx.ind.nxreg29, 4:0,4:0));
  NVWriteTV (pNv, 0x2c, SetBitField (r->nx.ind.nxreg2c, 5:0,5:0));
  NVWriteTV (pNv, 0x31, SetBitField (r->nx.ind.nxreg31,10:3,7:0));
  NVWriteTV (pNv, 0x35, Set8Bits (r->nx.ind.nxreg35));
  NVWriteTV (pNv, 0x3c, SetBitField (r->nx.ind.nxreg3c, 9:2,7:0));

  /* 0x06 has connector flags, but is not affected by it? */
  NVAndOrTV (pNv, 0x07, ~(SetBit(3)|SetBit(5)),
	       SetBitFlag (r->nx.ind.flags, NX_FLAG_CONN_3, 3)
	     | SetBitFlag (r->nx.ind.flags, NX_FLAG_CONN_5, 5));
  NVAndOrTV (pNv, 0x34, ~(SetBit(0)|SetBit(3)|SetBit(6)),
	       SetBitFlag (r->nx.ind.flags, NX_FLAG_SYS_0, 0)
	     | SetBitField (r->nx.ind.nxsys12,1:0,2:1)
	     | SetBitFlag (r->nx.ind.flags, NX_FLAG_SYS_3, 3)
	     | SetBitFlag (r->nx.ind.flags, NX_FLAG_SYS_6, 6));

  MMIO_OUT32 (pNv->riva.PTV, 0x204, 
	        SetBitField (r->nx.muxa,1:0,1:0)
	      | SetBitField (r->nx.muxb,1:0,5:4)
	      | SetBitField (r->nx.muxc,1:0,9:8)
	      | SetBitFlag (r->nx.flags1, NX_FLAG1_DACA, 12)
	      | SetBitFlag (r->nx.flags1, NX_FLAG1_DACB, 16)
	      | SetBitFlag (r->nx.flags1, NX_FLAG1_DACC, 20));
  MMIO_OUT32 (pNv->riva.PTV,0x304,SetBitField (r->nx.img.scaler_h,9:0,25:16));
  MMIO_OUT32 (pNv->riva.PTV,0x508,SetBitField (r->nx.img.scaler_v,9:0,25:16));
  MMIO_OUT32 (pNv->riva.PTV,0x208,SetBitField (r->nx.img.overscan,10:0,16:6));
  /* VIP: or 16:4 ?? */
  MMIO_OUT32 (pNv->riva.PTV,0x500,SetBitField (r->nx.img.vip1,15:0,19:4));
  MMIO_OUT32 (pNv->riva.PTV,0x504,SetBitField (r->nx.img.vip2,15:0,19:4));
  {
    CARD32 tmp = MMIO_IN32 (pNv->riva.PTV, 0x200);
    tmp &= ~MASKEXPAND(25:24);
    tmp |= SetBitField (r->nx.filter,1:0,25:24);
    MMIO_OUT32 (pNv->riva.PTV, 0x200, tmp);
  }
  for (i = 0; i < 7; i++)
  {
    MMIO_OUT32(pNv->riva.PTV,0x310+i*4,TVNxSetFiltCoeff(r->nx.filt_x1[0][i]));
    MMIO_OUT32(pNv->riva.PTV,0x32c+i*4,TVNxSetFiltCoeff(r->nx.filt_x1[1][i]));
    MMIO_OUT32(pNv->riva.PTV,0x350+i*4,TVNxSetFiltCoeff(r->nx.filt_x1[2][i]));
    MMIO_OUT32(pNv->riva.PTV,0x36c+i*4,TVNxSetFiltCoeff(r->nx.filt_x1[3][i]));

    MMIO_OUT32(pNv->riva.PTV,0x390+i*4,TVNxSetFiltCoeff(r->nx.filt_x2[0][i]));
    MMIO_OUT32(pNv->riva.PTV,0x3ac+i*4,TVNxSetFiltCoeff(r->nx.filt_x2[1][i]));
    MMIO_OUT32(pNv->riva.PTV,0x3d0+i*4,TVNxSetFiltCoeff(r->nx.filt_x2[2][i]));

    MMIO_OUT32(pNv->riva.PTV,0x3ec+i*4,TVNxSetFiltCoeff(r->nx.filt_x2[3][i]));
    MMIO_OUT32(pNv->riva.PTV,0x510+i*4,TVNxSetFiltCoeff(r->nx.filt_y [0][i]));
    MMIO_OUT32(pNv->riva.PTV,0x52c+i*4,TVNxSetFiltCoeff(r->nx.filt_y [1][i]));
    MMIO_OUT32(pNv->riva.PTV,0x550+i*4,TVNxSetFiltCoeff(r->nx.filt_y [2][i]));
    MMIO_OUT32(pNv->riva.PTV,0x56c+i*4,TVNxSetFiltCoeff(r->nx.filt_y [3][i]));
  }
/*

d600 ??
d604 system
d608 system

*/
}

void 
TVNxGetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  register NVPtr pNv = (NVPtr) this->ctrl;
  CARD8 regs[0x40];
  int i;

  RAISE (MSG_DEBUG, "tv get nx");
  for (i = 0x00; i < 0x40; i++) regs[i] = NVReadTV (pNv, i);
  r->nx.ind.fsub = (unsigned long) Set8Bits(regs[0x00]) << 24
                 | (unsigned long) Set8Bits(regs[0x01]) << 16
                 | (unsigned long) Set8Bits(regs[0x02]) << 8
                 | (unsigned long) Set8Bits(regs[0x03]);
  r->nx.ind.nxreg04 = Set8Bits    (regs[0x04]);
  r->nx.ind.nxreg08 = Set8Bits    (regs[0x08]);
  r->nx.ind.nxreg09 = SetBitField (regs[0x09],6:0, 6:0);
  r->nx.ind.nxreg0a = Set8Bits    (regs[0x0a]);	  
  r->nx.ind.nxreg0b = Set8Bits    (regs[0x0b]);	  
  r->nx.ind.nxreg0c = Set8Bits    (regs[0x0c]);	  
  r->nx.ind.nxreg0e = SetBitField (regs[0x0e],8:0,10:2);
  r->nx.ind.nxreg10 = SetBitField (regs[0x10],8:0,10:2);
  r->nx.ind.nxreg17 = SetBitField (regs[0x17],8:0,10:2);
  r->nx.ind.nxreg1e = SetBitField (regs[0x1e],8:0,10:2);
  r->nx.ind.nxreg20 = Set8Bits    (regs[0x20]);	  
  r->nx.ind.nxreg22 = Set8Bits    (regs[0x22]);	  
  r->nx.ind.nxreg29 = SetBitField (regs[0x29],4:0, 4:0);
  r->nx.ind.nxreg2c = SetBitField (regs[0x2c],5:0, 5:0);
  r->nx.ind.nxreg31 = SetBitField (regs[0x31],7:0,10:3);
  r->nx.ind.nxreg35 = SetBitField (regs[0x35],7:0, 7:0);
  r->nx.ind.nxreg3c = SetBitField (regs[0x3c],7:0, 9:2);

  r->nx.ind.flags = 0;
  {
    register CARD32 tmp = NVReadTV (pNv, 0x07);
    r->nx.ind.flags |= GetBitFlag (tmp, 3, NX_FLAG_CONN_3)
                    |  GetBitFlag (tmp, 5, NX_FLAG_CONN_5);
  }
  {
    register CARD32 tmp = NVReadTV (pNv, 0x34);
    r->nx.ind.flags |= GetBitFlag (tmp, 0, NX_FLAG_SYS_0)
                    |  GetBitFlag (tmp, 3, NX_FLAG_SYS_3)
                    |  GetBitFlag (tmp, 6, NX_FLAG_SYS_6);
    r->nx.ind.nxsys12 = SetBitField (tmp,2:1,1:0);
  }
  {
    register CARD32 tmp = MMIO_IN32 (pNv->riva.PTV, 0x200);
    r->nx.filter = SetBitField (tmp,25:24,1:0);
  }
  {
    register CARD32 tmp = MMIO_IN32 (pNv->riva.PTV, 0x204);
    r->nx.muxa = SetBitField (tmp,1:0,1:0);
    r->nx.muxb = SetBitField (tmp,5:4,1:0);
    r->nx.muxc = SetBitField (tmp,9:8,1:0);
    r->nx.flags1 = GetBitFlag (tmp,12,NX_FLAG1_DACA)
                 | GetBitFlag (tmp,16,NX_FLAG1_DACB)
                 | GetBitFlag (tmp,20,NX_FLAG1_DACC);
  }  
  r->nx.img.scaler_h = SetBitField(MMIO_IN32(pNv->riva.PTV,0x304),25:16, 9:0);
  r->nx.img.scaler_v = SetBitField(MMIO_IN32(pNv->riva.PTV,0x508),25:16, 9:0);
  r->nx.img.overscan = SetBitField(MMIO_IN32(pNv->riva.PTV,0x208), 16:6,10:0);
  r->nx.img.vip1 = MMIO_IN32(pNv->riva.PTV,0x500) >> 4;
  r->nx.img.vip2 = MMIO_IN32(pNv->riva.PTV,0x504) >> 4;
  for (i = 0; i < 7; i++)
  {
    r->nx.filt_x1[0][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x310+i*4));
    r->nx.filt_x1[1][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x32c+i*4));
    r->nx.filt_x1[2][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x350+i*4));
    r->nx.filt_x1[3][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x36c+i*4));

    r->nx.filt_x2[0][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x390+i*4));
    r->nx.filt_x2[1][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x3ac+i*4));
    r->nx.filt_x2[2][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x3d0+i*4));
    r->nx.filt_x2[3][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x3ec+i*4));

    r->nx.filt_y [0][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x510+i*4));
    r->nx.filt_y [1][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x52c+i*4));
    r->nx.filt_y [2][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x550+i*4));
    r->nx.filt_y [3][i]=TVNxGetFiltCoeff(MMIO_IN32(pNv->riva.PTV,0x56c+i*4));
  }
}

/* Also:
680608 bit 16 PWRDWN_DAC (always 2nd head???)
680630 bit 1-0 (on:2, off:0) (on head)
*/


void 
TVNxSetState (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register NVPtr pNv = (NVPtr) this->ctrl;

  switch (state)
  {
    case TV_OFF: 
    case TV_BARS: 
      NVWriteTV (pNv, 0x3e, 0x01);
      break;
    case TV_ON: 
      NVWriteTV (pNv, 0x3e, 0x00);
      break;
    default:
      break;
  }
}

long 
TVNxGetStatus (TVEncoderObj *this, int index)
{ 
  return 0;
}

TVConnect 
TVNxGetConnect (TVEncoderObj *this)
{
  /* not clear how to do that. Via testpoint?? */
  return CONNECT_BOTH; 
}

char*
TVDetectNx (TVChip *encoder)
{
  *encoder = TV_NVIDIA;
  return "NVIDIA internal (NV17)";
  /* FIXME check d22c bit 8: 0=NVIDIA-B, 1=NVIDIA-A */
}

TVEncoderObj tvNxTemplate = {
  type: TV_NVIDIA, ctrl: NULL, minClock: 10500, maxClock: 10500, 
  Create:     TVNxCreate,
  InitRegs:   TVNxInitRegs, 
  SetRegs:    TVNxSetRegs, 
  GetRegs:    TVNxGetRegs, 
  SetPort:    TVNxSetPort,
  GetPort:    TVNxGetPort,
  SetState:   TVNxSetState,
  GetConnect: TVNxGetConnect, 
  GetStatus:  TVNxGetStatus
};
