/*
Copyright (c) 2003, Dinesh Nadarajah
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, 
      this list of conditions and the following disclaimer in the documentation and/or 
      other materials provided with the distribution.
    * Neither the name of the author nor the names of its contributors 
      may be used to endorse or promote products derived from this software 
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "taxim-uni.h"

static const GtkIMContextInfo ta_uni_xim_info = { 
   "tamil99-uni",	    /* ID */
   N_("Tamil99 (UNICODE)"),   /* Human readable name */
   "gtk+",		    /* Translation domain */
   "",		    /* Dir for bindtextdomain (not strictly needed for "gtk+") */
   ""		            /* Languages for which this module is the default */
};

gint language = ENGLISH;
gint last_sent_keysym = 0;
guint32 last_sent_unicode = 0;

static const GtkIMContextInfo *info_list[] = {
  &ta_uni_xim_info
};

GType type_ta_uni = 0;

//-----------------------------------------------------------
void im_module_init (GTypeModule *type_module)
{
  gtk_im_context_ta_uni_xim_register_type (type_module);
}

//-----------------------------------------------------------
void im_module_exit (void)
{
  gtk_im_context_ta_uni_xim_shutdown ();
}

//-----------------------------------------------------------
void im_module_list (const GtkIMContextInfo ***contexts, int *n_contexts)
{
  *contexts = info_list;
  *n_contexts = G_N_ELEMENTS (info_list);
}

//-----------------------------------------------------------
GtkIMContext *im_module_create (const gchar *context_id)
{
  if (strcmp (context_id, "tamil99-uni") == 0)
     return GTK_IM_CONTEXT(g_object_new (type_ta_uni, NULL));
  else
     return NULL;
}

//-----------------------------------------------------------
void gtk_im_context_ta_uni_xim_shutdown (void)
{
}

//------------------------------------------------------------
void gtk_im_context_ta_uni_xim_register_type (GTypeModule *type_module)
{
  static const GTypeInfo im_context_ta_uni_xim_info =
  {
    sizeof (GtkIMContextClass),
    (GBaseInitFunc) NULL,
    (GBaseFinalizeFunc) NULL,
    (GClassInitFunc) gtk_im_context_ta_uni_xim_class_init,
    NULL,           /* class_finalize */    
    NULL,           /* class_data */
    sizeof (GtkIMContext),
    0,
    (GtkObjectInitFunc) gtk_im_context_ta_uni_xim_init,
  };

  type_ta_uni = 
    g_type_module_register_type (type_module,
				 GTK_TYPE_IM_CONTEXT,
				 "GtkIMContextTA99UNI",
				 &im_context_ta_uni_xim_info, 0);
}

//------------------------------------------------------------
void gtk_im_context_ta_uni_xim_class_init (GtkIMContextClass *class)
{
  GtkIMContextClass *im_context_class = GTK_IM_CONTEXT_CLASS (class);

  im_context_class->filter_keypress = gtk_im_context_ta_uni_xim_filter_keypress;
}

//------------------------------------------------------------
void gtk_im_context_ta_uni_xim_init (GtkIMContext *im_context_xim)
{
  gtk_im_context_set_use_preedit(im_context_xim, FALSE);
}

//------------------------------------------------------------
gboolean gtk_im_context_ta_uni_xim_filter_keypress (GtkIMContext *context,
				                    GdkEventKey  *event)
{
   if (event->type == GDK_KEY_PRESS) {

      if (event->keyval == GDK_BackSpace)
         last_sent_unicode = 0;         

      if ((event->state & GDK_SHIFT_MASK) &&
         (event->keyval == GDK_Shift_L || event->keyval == GDK_Shift_R)) {
             // switch language codes
             language = !language;
             return TRUE;
      }

      else if (language == TAMIL &&
               event->keyval >= GDK_space && event->keyval <= GDK_asciitilde) {
                  return process_unicode_tamil99(context, event);
      }

      else {
             guint32 u;
	     guchar utf8[16];

	     u = gdk_keyval_to_unicode(event->keyval);
	     if (u == 0) {
		 return FALSE;
	     }
	     utf8[g_unichar_to_utf8(u, utf8)] = '\0';
             g_signal_emit_by_name(context, "commit", utf8);
             last_sent_keysym = event->keyval;
             last_sent_unicode = u;
             return TRUE;
      }

   }
   
   return FALSE;
}

//----------------------------------------------------------------
void send_1_unicode_value(GtkIMContext *context, guint32 u1) {

   guchar utf8[16]; 
   
   utf8[g_unichar_to_utf8(u1, utf8)] = '\0';

   g_signal_emit_by_name(context, "commit", utf8);        
}

//----------------------------------------------------------------
void send_2_unicode_value(GtkIMContext *context, guint32 u1, guint32 u2) {

   guchar utf8[16]; 

   utf8[g_unichar_to_utf8(u1, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);  
   utf8[g_unichar_to_utf8(u2, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);        
}

//----------------------------------------------------------------
void send_3_unicode_value(GtkIMContext *context, guint32 u1, guint32 u2, guint32 u3) {

   guchar utf8[16]; 

   utf8[g_unichar_to_utf8(u1, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);  
   utf8[g_unichar_to_utf8(u2, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);
   utf8[g_unichar_to_utf8(u3, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);        
}

//----------------------------------------------------------------
void send_4_unicode_value(GtkIMContext *context, guint32 u1, guint32 u2, 
			  guint32 u3, guint32 u4) {

   guchar utf8[16]; 

   utf8[g_unichar_to_utf8(u1, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);  
   utf8[g_unichar_to_utf8(u2, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);
   utf8[g_unichar_to_utf8(u3, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8); 
   utf8[g_unichar_to_utf8(u4, utf8)] = '\0';
   g_signal_emit_by_name(context, "commit", utf8);       
}

//----------------------------------------------------------------
gboolean process_unicode_tamil99(GtkIMContext *context, GdkEventKey *event) {

   guint32 u;

   switch (event->keyval) {

      case GDK_y:
         if (last_sent_unicode == 0xbb3) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb3);  
            last_sent_unicode = 0xbb3;          
         }
         break;

      case GDK_u:
         if (last_sent_unicode == 0xbb1) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb1);  
            last_sent_unicode = 0xbb1;          
         }
         break;

      case GDK_i:
         if (last_sent_unicode == 0xba9) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xba9);  
            last_sent_unicode = 0xba9;          
         }
         break;

      case GDK_o:
         if (last_sent_unicode == 0xb9f) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb9f);  
            last_sent_unicode = 0xb9f;          
         }
         break;

      case GDK_p:
         if (last_sent_unicode == 0xba3) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xba3);  
            last_sent_unicode = 0xba3;          
         }
         break;

      case GDK_bracketleft:
         if (last_sent_unicode == 0xb9a) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb9a);  
            last_sent_unicode = 0xb9a;          
         }
         break;

      case GDK_bracketright:
         if (last_sent_unicode == 0xb9e) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb9e);  
            last_sent_unicode = 0xb9e;          
         }
         break;

      case GDK_h:
         if (last_sent_unicode == 0xb95) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb95);  
            last_sent_unicode = 0xb95;          
         }
         break;

      case GDK_j:
         if (last_sent_unicode == 0xbaa) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbaa);  
            last_sent_unicode = 0xbaa;          
         }
         break;

      case GDK_k:
         if (last_sent_unicode == 0xbae) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbae);  
            last_sent_unicode = 0xbae;          
         }
         break;

      case GDK_l:
         if (last_sent_unicode == 0xba4) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xba4);  
            last_sent_unicode = 0xba4;          
         }
         break;

      case GDK_semicolon:
         if (last_sent_unicode == 0xba8) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xba8);  
            last_sent_unicode = 0xba8;          
         }
         break;

      case GDK_apostrophe:
         if (last_sent_unicode == 0xbaf) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbaf);  
            last_sent_unicode = 0xbaf;          
         }
         break;

      case GDK_v:
         if (last_sent_unicode == 0xbb5) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb5);  
            last_sent_unicode = 0xbb5;          
         }
         break;

      case GDK_b:
         if (last_sent_unicode == 0xb99) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb99);  
            last_sent_unicode = 0xb99;          
         }
         break;

      case GDK_n:
         if (last_sent_unicode == 0xbb2) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb2);  
            last_sent_unicode = 0xbb2;          
         }
         break;

      case GDK_m:
         if (last_sent_unicode == 0xbb0) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb0);  
            last_sent_unicode = 0xbb0;          
         }
         break;

      case GDK_slash:
         if (last_sent_unicode == 0xbb4) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb4);  
            last_sent_unicode = 0xbb4;          
         }
         break;

      case GDK_Q:
         if (last_sent_unicode == 0xbb8) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb8);  
            last_sent_unicode = 0xbb8;          
         }
         break;

      case GDK_W:
         if (last_sent_unicode == 0xbb7) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb7);  
            last_sent_unicode = 0xbb7;          
         }
         break;

      case GDK_E:
         if (last_sent_unicode == 0xb9c) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb9c);  
            last_sent_unicode = 0xb9c;          
         }
         break;

      case GDK_R:
         if (last_sent_unicode == 0xbb9) {
            send_1_unicode_value(context, 0xbcd);
            last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xbb9);  
            last_sent_unicode = 0xbb9;          
         }
         break;

      case GDK_T:
         if (last_sent_unicode == 0x87) {
            gtk_im_context_delete_surrounding (context, -1, 1);
            send_1_unicode_value(context, 0x8c);
            last_sent_unicode = 0x8c;
         }
         else {
            send_1_unicode_value(context, 0x87);  
            last_sent_unicode = 0x87;          
         }
         break;

      case GDK_Y:         
         send_4_unicode_value(context, 0xbb8, 0xbcd, 0xbb0, 0xbc0);  
         last_sent_unicode = 0xbc0;          
         break;

      case GDK_a:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               last_sent_unicode = 0;
         }
         else {
            send_1_unicode_value(context, 0xb85);
            last_sent_unicode = 0xb85;
         }
         break;

      case GDK_q:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbbe);
               last_sent_unicode = 0xbbe;
         }
         else {
            send_1_unicode_value(context, 0xb86);
            last_sent_unicode = 0xb86;
         }
         break;

      case GDK_s:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
                  send_1_unicode_value(context, 0xbbf);
                  last_sent_unicode = 0xbbf;
         }
         else {
            send_1_unicode_value(context, 0xb87);
            last_sent_unicode = 0xb87;
         }
         break;

      case GDK_w:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
                  send_1_unicode_value(context, 0xbc0);
                  last_sent_unicode = 0xbc0;
         }
         else {
            send_1_unicode_value(context, 0xb88);
            last_sent_unicode = 0xb88;
         }
         break;

      case GDK_r:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbc8);
               last_sent_unicode = 0xbc8;
         }
         else {
            send_1_unicode_value(context, 0xb90);
            last_sent_unicode = 0xb90;
         }
         break;

      case GDK_g:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbc6);
               last_sent_unicode = 0xbc6;
         }
         else {
            send_1_unicode_value(context, 0xb8e);
            last_sent_unicode = 0xb8e;
         }
         break;

      case GDK_t:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbc7);
               last_sent_unicode = 0xbc7;
         }
         else {
            send_1_unicode_value(context, 0xb8f);
            last_sent_unicode = 0xb8f;
         }
         break;

      case GDK_c:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbca);
               last_sent_unicode = 0xbca;
         }
         else {
            send_1_unicode_value(context, 0xb92);
            last_sent_unicode = 0xb92;
         }
         break;

      case GDK_x:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbcb);
               last_sent_unicode = 0xbcb;
         }
         else {
            send_1_unicode_value(context, 0xb93);
            last_sent_unicode = 0xb93;
         }
         break;

      case GDK_z:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbcc);
               last_sent_unicode = 0xbcc;
         }
         else {
            send_1_unicode_value(context, 0xb94);
            last_sent_unicode = 0xb94;
         }
         break;

      case GDK_f:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbcd);
               last_sent_unicode = 0xbcd;
         }
         else {
            send_1_unicode_value(context, 0xb83);
            last_sent_unicode = 0xb83;
         }
         break;

      case GDK_d:

         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbc1);
               last_sent_unicode = 0xbc1;  
         }         
         else {
            send_1_unicode_value(context, 0xb89);
            last_sent_unicode = 0xb89;
         }
         break;

      case GDK_e:
         if (last_sent_unicode >= 0xb95 && last_sent_unicode <= 0xbb9) {
               send_1_unicode_value(context, 0xbc2);
               last_sent_unicode = 0xbc2;  
         }         
         else {
            send_1_unicode_value(context, 0xb8a);
            last_sent_unicode = 0xb8a;
         }
         break;

      case GDK_O:
         send_1_unicode_value(context, 0x5b);  
         last_sent_unicode = 0x5b;          
         break;

      case GDK_P:
         send_1_unicode_value(context, 0x5d);  
         last_sent_unicode = 0x5d;          
         break;

      case GDK_M:
         send_1_unicode_value(context, 0x2f);  
         last_sent_unicode = 0x2f;          
         break;

      case GDK_K:
         send_1_unicode_value(context, 0x22);  
         last_sent_unicode = 0x22;          
         break;

      case GDK_L:
         send_1_unicode_value(context, 0x3a);  
         last_sent_unicode = 0x3a;          
         break;

      case GDK_colon:
         send_1_unicode_value(context, 0x3b);  
         last_sent_unicode = 0x3b;          
         break;

      case GDK_quotedbl:
         send_1_unicode_value(context, 0x27);  
         last_sent_unicode = 0x27;          
         break;

      default:

         u = gdk_keyval_to_unicode(event->keyval);
         if (u == 0) {
            return FALSE;
         }
         send_1_unicode_value(context, u);
         last_sent_keysym = event->keyval;
         last_sent_unicode = u;
         break;
   }
   
   return TRUE;
}


